/*
  heuristics for MAX SAT
*/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include "cpu_time.c"

/****** macros ***************************************************************/
#define ABS(x)  (((x)>0) ? (x) : -(x))

/****** constants ************************************************************/
#define FALSE		0
#define TRUE		!FALSE
#define BITSFORFLIP	3
#define MASK		(1 << BITSFORFLIP) - 1
#define MAXFLIP		3
#define HASHWOOF	0

/****** output definitions ***************************************************/
#define DBG		1	/* debug or not (1/0) */
#define LEVEL		1	/* output level (0,1,2,3) */
#define OUTELO		0	/* output every local optimal value */
#define PRINT_INPUT	0	/* output input or not (1/0) */
#define REDUNDANTCHK	0	/* check redundancy or not (1/0) */
#define OUTFOREVERY	50000	/* output for every OUTFOREVERY samples */
#define EVALBEST	1	/* re-evaluate bestsol for debug */
#define OUTBESTSOL	0	/* output bestsol or not (1/0)
				   OUTBESTSOL=1 -> EVALBEST should be 1*/

/****** algorithmic parameters (changable by inputs) *************************/
#define ITRY		0	/* number of trials */
#define MAXMEMACC	0	/* number of memory accesses */
#define MAXMOVES	0	/* number of moves allowed */
#define MAXSAMPLES	1000000000	/* number of samples allowed */
				/* priority: MAXMEMACC -> ITRY ->
				             MAXMOVES -> MAXSAMPLES */
#define TIMELIM		3600	/* maximum computational time (secs.) */
#define INITREV		0	/* useage of reversed sol for init (0/1/2) */
#define FLIPNUM		1	/* neighbor def: Humming dist. <= FLIPNUM */
#define FIRST		1	/* neighbor search (BEST -> 0; FIRST -> 1) */
#define CHTOBEST	0	/* BEST is checked near local opt */
#define PLUSCHK		-1	/* scan only the flip cands with modify>0 */
#define ZEROCHK		1	/* at least one literal is included in zerocl*/
#define RINGORDERED	0	/* mdfy_ring is ordered in inc. order of fln */
#define OPTCHK		1	/* if optimal is found then exit (0/1) */
#define MALLOCUNIT	1024	/* unit for allocate memory for modify */
#define HASHSIZE	19	/* 2^HASHSIZE is used for hash size */
#define RMZEROMEM	0	/* rm mem space with mdfy_val=0 or not (0/1) */
#define SEED		13	/* default value of random seed */
#define LOGOUTPUT	1	/* logarithmic output or not */

static int  itry, maxsamples, outforevery, mallocunit, hashsize, maxmoves,
            maxmemacc, timelim;
static char initrev, flipnum, first, outelo, pluschk, rmzeromem, optchk,
            ringordered, chtobest, zerochk, outbestsol, logoutput;

struct modify {
  char fln;                   /* number of flips */
  int  flip[MAXFLIP];         /* list of flipped variables */
  int  val;                   /* modified value */
  int  combpt;                /* used when pluschk=2,4 */
  int  ntpt;                  /* used when pluschk=3,4 */
  char inring;                /* indicate inclusion in the candidate ring */
  struct modify *hnext;       /* pointer to the next     (hash) */
  struct modify *rprev;       /* pointer to the previous (candidate ring) */
  struct modify *rnext;       /* pointer to the next     (candidate ring) */
  struct modify *fl2_prev[2]; /* pointer to the previous (fl2_list) */
  struct modify *fl2_next[2]; /* pointer to the next     (fl2_list) */
};

static int m, n, optimal, bestgain, isamples, imoves, *w, *nlit, **lit,
           *nclause[2], **clause[2], *ntrue, *ntrue_rev, cgain, cgain_rev,
           nbsize, *flcand, ntry, *ntrue_prev, nzeroclause, org_n,
           nmem_heads, mem_heads_size, nfree_mems, free_mems_size,
           hashmask, nmodify_ring, nmem_used, max_nmem_used, nhashplus,
           max_hash_trace, *clchnum, *clchplus, *hashplus, *ref_hashplus,
           nmodify_ring_f[MAXFLIP], *nfl2_list, *inzeroclause,
           max_var_appear, max_clause_size;
static char optknown, *csol, *csol_rev, *bestsol, optfound;
static double starttime, sum_max_nmem_used, sum_max_hash_trace, 
              nmodify_ring_sum, avr_var_appear, avr_clause_size,
              sum_memory_access_calc, sum_memory_access_update,
              sum_gain;
static unsigned long x_n, *nbpointer, seed;
static struct modify **mem_heads, **free_mems, **hash, *modify_init,
                     *modify_init_f[MAXFLIP], **fl2_list;

/*************************** functions ***************************************/
int  rand_from(int min, int max);
void read_parameters(int argc, char *arcv[]);
void read_data(void);
void rm_redundancy(void);
void random_init(void);
int  evaluate(char *sol, int *nt);
void improve_csol(void);
int  i_choose_j(int i, int j);
void initialize_nbpointer(void);
void randomize_nbpointer(void);
void free_memories(void);
void check_nbsize(void);
void decode_nbpointer(int pt, int *iflip, int *ii);
int  flipgain(int iflip, int *ii, int *ii1);
void swap_csol_and_csol_rev(void);
void update_bestsol(void);
void output_current_state(void);
void output_current_state_m(void);
void output_parameters(void);
void output_results(void);
void update_ntrue(int iflip, int *ii);
void init_mems(void);
void make_nb_order_list(void);
void initialize_modify(void);
void add_to_modify(int npos, int nneg, int *pos, int *neg, int w_i,
		   char addsign);
void free_mdfy_mems(void);
void fl2_add(struct modify *mdfy);
void fl2_delete(struct modify *mdfy);
void add_to_hash(char iflip, int *ii, int mdfy_val, int *hashvl, int *mdfy_vp,
		 struct modify **mdfyp, struct modify **mdfyprevp); 
void add_to_ring(int hashvalue, struct modify *mdfy, struct modify *mdfyprev); 
void init_inzeroclause(void);
int  included_in_zerocl(int iflip, int *ii);
void output_HJ_lb(void);
int  gcd00(int aa, int bb);
void *malloc_e(size_t size);

/***** copy and read the parameters ******************************************/
void read_parameters(int argc, char *argv[])
{
  int i;
  unsigned long h;

  /**** copy the parameters ****/
  outforevery = OUTFOREVERY;
  itry        = ITRY;
  maxsamples  = MAXSAMPLES;
  maxmoves    = MAXMOVES;
  initrev     = INITREV;
  flipnum     = FLIPNUM;
  first       = FIRST;
  chtobest    = CHTOBEST;
  outelo      = OUTELO;
  pluschk     = PLUSCHK;
  zerochk     = ZEROCHK;
  ringordered = RINGORDERED;
  optchk      = OPTCHK;
  mallocunit  = MALLOCUNIT;
  hashsize    = HASHSIZE;
  rmzeromem   = RMZEROMEM;
  outbestsol  = OUTBESTSOL;
  maxmemacc   = MAXMEMACC;
  seed        = SEED;
  logoutput   = LOGOUTPUT;
  timelim     = TIMELIM;

  /**** read the parameters ****/
  if(argc>0 && (argc % 2)==0){
    printf("USAGE: mls [param_name, param_value] [name, value]...\n");
    exit(1);}
  else{
    for(i=1;i<argc;i+=2){
      if(strcmp(argv[i],"outforevery")==0) outforevery=atoi(argv[i+1]);
      if(strcmp(argv[i],"itry")==0) itry=atoi(argv[i+1]);
      if(strcmp(argv[i],"maxsamples")==0) maxsamples=atoi(argv[i+1]);
      if(strcmp(argv[i],"maxmoves")==0) maxmoves=atoi(argv[i+1]);
      if(strcmp(argv[i],"initrev")==0) initrev=atoi(argv[i+1]);
      if(strcmp(argv[i],"flipnum")==0) flipnum=atoi(argv[i+1]);
      if(strcmp(argv[i],"first")==0) first=atoi(argv[i+1]);
      if(strcmp(argv[i],"chtobest")==0) chtobest=atoi(argv[i+1]);
      if(strcmp(argv[i],"outelo")==0) outelo=atoi(argv[i+1]);
      if(strcmp(argv[i],"pluschk")==0) pluschk=atoi(argv[i+1]);
      if(strcmp(argv[i],"zerochk")==0) zerochk=atoi(argv[i+1]);
      if(strcmp(argv[i],"ringordered")==0) ringordered=atoi(argv[i+1]);
      if(strcmp(argv[i],"optchk")==0) optchk=atoi(argv[i+1]);
      if(strcmp(argv[i],"mallocunit")==0) mallocunit=atoi(argv[i+1]);
      if(strcmp(argv[i],"hashsize")==0) hashsize=atoi(argv[i+1]);
      if(strcmp(argv[i],"rmzeromem")==0) rmzeromem=atoi(argv[i+1]);
      if(strcmp(argv[i],"outbestsol")==0) outbestsol=atoi(argv[i+1]);
      if(strcmp(argv[i],"maxmemacc")==0) maxmemacc=atoi(argv[i+1]);
      if(strcmp(argv[i],"seed")==0) seed=atoi(argv[i+1]);
      if(strcmp(argv[i],"logoutput")==0) logoutput=atoi(argv[i+1]);
      if(strcmp(argv[i],"timelim")==0) timelim=atoi(argv[i+1]);
    }
  }
  /**** check the validity of some parameters ****/
  if(pluschk==-1){
    switch(flipnum){
    case  1: pluschk = 1; break;
    case  2: pluschk = 1; break;
    case  3: pluschk = 2; break;
    default: pluschk = 0; break;
    }
  }
  if(logoutput==1) outforevery = 1;
  if(outbestsol == 1 && EVALBEST == 0){
    printf("#\n# To output the best solution, ");
    printf("please set EVALBEST=1 and re-compile!!!\n#\n");
    outbestsol = 0;}
  if(pluschk==0 /*|| pluschk==2 || pluschk==4*/) ringordered = 0;
  if(pluschk==2 || pluschk==4){ flipnum = 3; /*rmzeromem = 0;*/}
  if(first==0 && chtobest==1) chtobest = 0;
  if(pluschk==0 && flipnum >= (1<<BITSFORFLIP)){
    printf("WARNING: 'flipnum' should be at most 2^BITSFORFLIP -1.\n");
    printf("         To use larger flipnum, enlarge BITSFORFLIP.\n");
    exit(1);}
  if(flipnum > MAXFLIP){
    printf("WARNING: 'flipnum' should be at most MAXFLIP.\n");
    printf("         To use larger flipnum, enlarge MAXFLIP.\n");
    exit(1);}
  if((int)(log(INT_MAX)/log(2)+1) < hashsize){
    printf("WARNING: 'hashsize' should be at most log_2(INT_MAX)\n");
    exit(1);}
#if HASHWOOF==1
  h = (1 << hashsize) -1;
  if(INT_MAX/h < h){
    printf("WARNING: '(2^hashsize)^2' should be at most INT_MAX\n");
    exit(1);}
#endif
}

/***** output parameters and problem size ************************************/
void output_parameters(void)
{
  printf("# %d variables, %d clauses",org_n,m);
  if(optknown==1)printf(", optimal = %d",optimal);
  if(max_var_appear!=0){
    printf("\n# clause_size: max = %d, avr. = %g;",
	   max_clause_size, avr_clause_size);
    printf("  var_appear: max = %d, avr. = %g",
	   max_var_appear, avr_var_appear);}
  printf("\n# ");
  if(maxmemacc>0)printf("%d mem accesses",maxmemacc);
  else if(itry>0)printf("%d trials",itry);
  else if(maxmoves>0)printf("%d moves",maxmoves);
  else printf("%d samples",maxsamples);
  printf(" allowed (time limit %d secs.), initrev: %d", timelim, initrev);
  if(initrev==1)printf(" (better of csol & csol_rev)");
  if(initrev==2)printf(" (both csol & csol_rev)");
  printf(", %d flips,",flipnum);
  if(pluschk>=1)printf(" pluschk: %d,",pluschk);
  if(zerochk==1)printf(" zerochk");
  if(ringordered==1)printf("(ring ordered)");
  printf(",");
  if(optchk==1)printf(" opt check,");
  if(rmzeromem==1)printf(" rm zero mem,");
  printf(" mallocunit: %d,", mallocunit);
  printf(" hashsize: %d,", 1<<hashsize);
  if(first==1)printf(" FIRST");
  else printf(" BEST");
  if(chtobest==1)printf("(chtobest)");
  printf(", seed: %d", seed);
  printf("\n");
}

/***** improve 'csol' by FIRST ***********************************************/
void improve_csol(void)
{
  int i, j, counter, pt, *ii, *ii1, iflip, gaininc, *selected_ii, ringpt,
      selected_iflip, selected_gaininc, gaininc2, sum0, sum1, rrr, ringcnt,
      *prev_selected_ii, prev_selected_iflip, prev_selected_gaininc;
  char *mark, improve_found, bestchanged, exit_yes;

  if(pluschk==0){if((ii=(int *)malloc(flipnum*sizeof(int)+1))==NULL) exit(1);}
  if((selected_ii=(int *)malloc(2*flipnum*sizeof(int)))==NULL) exit(1);
  prev_selected_ii = selected_ii + flipnum;
  if((ii1=(int *)malloc((flipnum+1)*sizeof(int)))==NULL) exit(1);
  if(first==1 && pluschk==0) randomize_nbpointer();
  initialize_modify();
  nzeroclause=0;
  if(optchk==1 && zerochk==0)for(i=0; i<m; i++)if(ntrue[i]==0)nzeroclause++;
  if(zerochk==1)init_inzeroclause();
      /* zerochk=1 -> nzeroclause is also initialized in this subroutine */
  exit_yes = 0; prev_selected_iflip = -1;
  pt=0;
  do{                                        /**** outer loop ****/
    nmodify_ring_sum += nmodify_ring;
    if(optchk==1) if(nzeroclause==0){         /* all clauses are satisfied */
	optfound=1;if(outelo==0)return;}
    if(pluschk>=1) if(nmodify_ring==0) break; /* locally optimal */
    selected_gaininc = -INT_MAX;
    bestchanged = 0;
    ringpt = ringcnt = 0;
    while(nmodify_ring_f[ringpt]==0){
      ringpt++;
      if(ringpt >= flipnum) break;
    }
    /*do{rrr = rand_from(1,nmodify_ring - 1);}
    while(gcd00(nmodify_ring,rrr)!=1);/**/
    /*printf("rrr = %d  nmodify_ring = %d\n",rrr,nmodify_ring);/**/
    for(counter=0; (pluschk==0 && counter<nbsize) ||
	(pluschk>=1 && counter<nmodify_ring); counter++){/**** inner loop ****/
      if(pluschk==0){
	decode_nbpointer(pt,&iflip,ii);
	pt = (pt + 1) % nbsize;
      }
      else if(ringordered==0){
	iflip = modify_init->fln;
	ii = (modify_init->flip);
	modify_init = modify_init->rnext;/**/
	/*for(i=0; i<rrr; i++) modify_init = modify_init->rnext;/**/
      }
      else{/* pluschk >= 1 & ringordered = 1 */
	iflip = modify_init_f[ringpt]->fln;
	ii = (modify_init_f[ringpt]->flip);
	modify_init_f[ringpt] = modify_init_f[ringpt]->rnext;
#if FALSE /* display */
	printf("iflip = %2d: ", iflip);
	for(i=0; i<iflip; i++)printf("%3d", ii[i]+1);
	printf("\n");
#endif
	ringcnt++;
	if(ringcnt >= nmodify_ring_f[ringpt]){
	  ringcnt = 0;
	  for(;;){
	    ringpt++;
	    if(ringpt >= flipnum) break;
	    if(nmodify_ring_f[ringpt]>0) break;
	  }
	}
      }
      if(zerochk==1) if(included_in_zerocl(iflip,ii)==0) continue;
      gaininc = flipgain(iflip,ii,ii1);
      isamples++;
      if(gaininc>selected_gaininc){
	selected_gaininc = gaininc;
	selected_iflip   = iflip;
	for(i=0;i<iflip;i++) selected_ii[i] = ii[i];
	if(cgain+gaininc > bestgain){bestgain=cgain+gaininc;bestchanged=1;}
	if(first==1)if(gaininc>0){ /* FIRST */
	  if(maxmoves==0) output_current_state();
	  break;}
      }
      if(maxmoves==0) output_current_state();
    }    /**** end of inner loop ****/
    if(selected_gaininc>0){
      cgain += selected_gaininc;
      update_ntrue(selected_iflip,selected_ii);
      for(i=0;i<selected_iflip;i++)csol[selected_ii[i]]=1-csol[selected_ii[i]];
      imoves++;
      if(bestchanged==1)update_bestsol();
      if(maxmoves>0) output_current_state_m();
      prev_selected_gaininc = selected_gaininc;
      prev_selected_iflip = selected_iflip;
      for(i=0;i<selected_iflip;i++) prev_selected_ii[i] = selected_ii[i];

      /*if(first==0)printf("BESTMOVE %5d\n",ntry);/**/
      /*printf("\n");/**/

      if((optknown==1 && bestgain==optimal) || (optchk==1 && nzeroclause==0)){
	optfound=1;if(optchk==1&&outelo==0)return;}
#if LEVEL >= 2 /**** output every move ****/
      printf("sample%8d move%6d[best=%7d, cgain=%7d]:",
	     isamples,imoves,bestgain,cgain);
#if LEVEL >= 3
      if((mark=(char *)malloc(n*sizeof(char)))==NULL) exit(1);
      for(i=0;i<n;i++)mark[i]=0;
      for(i=0;i<selected_iflip;i++)mark[selected_ii[i]]=1;
      for(i=0;i<n;i++){
	if(mark[i]==1)printf("%1d*",csol[i]);
	else printf("%1d ",csol[i]);}
      free(mark);
#endif
#if TRUE /**** output number of clauses with ntrue=0 or 1 ****/
      sum0 = sum1 = 0;
      for(i=0;i<m;i++){
	if(ntrue[i]==0)      sum0++;
	else if(ntrue[i]==1) sum1++;}
      printf(" #clauses with ntrue=0: %3d; ntrue=1: %3d; ntrue=0,1: %3d",
	     sum0, sum1, sum0+sum1);
#endif
      printf("\n");
#endif        /**** end of output ****/
    }
    if(selected_gaininc <= 0 || (pluschk>=1 && nmodify_ring==0)){
      /*printf("!!!first = %d, chtobest = %d\n",first,chtobest);/**/
      if(chtobest==0 || first==0 || prev_selected_iflip==(-1)) exit_yes = 1;
      else{
	first = 0;
	cgain -= prev_selected_gaininc;
	update_ntrue(prev_selected_iflip,prev_selected_ii);
	for(i=0;i<prev_selected_iflip;i++)
	  csol[prev_selected_ii[i]]=1-csol[prev_selected_ii[i]];
	imoves++;
	if(maxmoves>0) output_current_state_m();
	/*printf("nmodify_ring = %d\n", nmodify_ring);/**/
      }
    }
  }while(exit_yes==0); /**** end of outer loop ****/
  if(chtobest==1) first = 1;
  if(pluschk==0){free(ii);} free(selected_ii); free(ii1);
  if(outelo==1){
    printf("#local opt: %8d", cgain);
    if(optknown==1)
      printf("  (%9.5f%%)", (double)(optimal-cgain)*100.0/optimal);
    printf("\n");
  }
  free_mdfy_mems();
}

/***** update ntrue and modify **********************************/
void update_ntrue(int iflip, int *ii)
{
  int i, j, nclause_t_i, nclause_f_i, cltemp, iii, nclchplus, temp,
      *variables, npos, nneg, nlit_cltemp, littemp;
  char truelit, falselit;
  struct modify *mdfy;

  nclchplus = 0;
  /* store the flips for each clause and update ntrue */
  for(i=0;i<iflip;i++){
    iii=ii[i];
    truelit = csol[iii];
    falselit = 1 - truelit;
    nclause_t_i = nclause[truelit][iii];
    nclause_f_i = nclause[falselit][iii];
    for(j=0;j<nclause_t_i;j++){
      cltemp = clause[truelit][iii][j];
      if(clchnum[cltemp]==0){
	ntrue_prev[cltemp] = ntrue[cltemp];
	clchplus[nclchplus++] = cltemp;
	clchnum[cltemp]=1; }
      ntrue[cltemp]++;/**/
    }
    for(j=0;j<nclause_f_i;j++){
      cltemp = clause[falselit][iii][j];
      if(clchnum[cltemp]==0){
	ntrue_prev[cltemp] = ntrue[cltemp];
	clchplus[nclchplus++] = cltemp;
      	clchnum[cltemp]=1;}
      ntrue[cltemp]--;/**/
    }
  }
#if FALSE
  for(i=0; i<nclchplus; i++){
    printf("clause[%3d]: %2d: ntrue %2d->%2d:",
	   clchplus[i]+1, clchnum[clchplus[i]],
	   ntrue_prev[clchplus[i]], ntrue[clchplus[i]]);
    printf("\n"); }
#endif
  /**** update ntrue and modify list ****/
  if((variables=(int *)malloc(n*sizeof(int)))==NULL) exit(1);
  for(i=0; i<nclchplus; i++){
    cltemp = clchplus[i];
    nlit_cltemp = nlit[cltemp];
    /* modify the number of zero_clause */
    if(optchk==1 || zerochk==1){
      if(ntrue_prev[cltemp]==0 && ntrue[cltemp]>0){
	nzeroclause--;
	if(zerochk==1){
	  for(j=0; j<nlit_cltemp; j++){
	    littemp = lit[cltemp][j];
	    littemp = ABS(littemp) - 1;
	    inzeroclause[littemp]--;
	  }
	}
      }
      else if(ntrue_prev[cltemp]>0 && ntrue[cltemp]==0){
	nzeroclause++;
	if(zerochk==1){
	  for(j=0; j<nlit_cltemp; j++){
	    littemp = lit[cltemp][j];
	    littemp = ABS(littemp) - 1;
	    inzeroclause[littemp]++;
	  }
	}
      }
    }
    /* clear the previous modify value */
    if(ntrue_prev[cltemp] <= flipnum){
      npos = nneg = 0;
      for(j=0; j<nlit_cltemp; j++){
	littemp = lit[cltemp][j];
	if(littemp > 0){
	  littemp--;
	  if(csol[littemp]==1) variables[npos++]         = littemp;
	  else                 variables[flipnum+nneg++] = littemp;
	}
	else{
	  littemp = -littemp-1;
	  if(csol[littemp]==0) variables[npos++]         = littemp;
	  else                 variables[flipnum+nneg++] = littemp;
	}
      }
      add_to_modify(npos,nneg,variables,variables+flipnum,w[cltemp],-1);
    }
    /* update modify value */
    if(ntrue[cltemp] <= flipnum){
      for(j=0; j<iflip; j++) csol[ii[j]] = 1 - csol[ii[j]];
      npos = nneg = 0;
      for(j=0; j<nlit_cltemp; j++){
	littemp = lit[cltemp][j];
	if(littemp > 0){
	  littemp--;
	  if(csol[littemp]==1) variables[npos++]         = littemp;
	  else                 variables[flipnum+nneg++] = littemp;
	}
	else{
	  littemp = -littemp-1;
	  if(csol[littemp]==0) variables[npos++]         = littemp;
	  else                 variables[flipnum+nneg++] = littemp;
	}
      }
      add_to_modify(npos,nneg,variables,variables+flipnum,w[cltemp],1);
      for(j=0; j<iflip; j++) csol[ii[j]] = 1 - csol[ii[j]];
    }
  }
  /* clear clchnum list */
  for(i=0; i<nclchplus; i++) clchnum[clchplus[i]] = 0;
#if FALSE
  printf("clchnum: ");for(i=0;i<m;i++)printf("%d",clchnum[i]);printf("\n");
#endif
  free(variables);
#if FALSE /* display modify ring list */
  printf("#zeroclause = %d ", nzeroclause);/**/
  if(TRUE){ /* count the number of plus cells */
    int counter_p, counter_m, counter_0;
    counter_p = counter_m = counter_0 = 0;
    for(i=0; i<nhashplus; i++){
      mdfy = hash[hashplus[i]];
      for(;;){
	if(mdfy->val > 0) counter_p++;
	else if(mdfy->val < 0) counter_m++;
	else counter_0++;
	if(mdfy->hnext == NULL) break;
	else mdfy = mdfy->hnext;
      }
    }
    printf("#plus cells = %d #zero cells = %d #minus cells = %d\n",
	   counter_p, counter_0, counter_m);
  }
  if(pluschk>=1 && ringordered==0){
    printf("modify ring list (%d) ([mdfy_val,iflip(flips),ntpt,combpt]:\n",
	   nmodify_ring);
    mdfy = modify_init;
    for(i=0; i<nmodify_ring; i++){
      printf("->[%d,%d(", mdfy->val, mdfy->fln);
      for(j=0; j<mdfy->fln; j++){
	if(j!=0)printf(",");printf("%d",mdfy->flip[j]+1);}
      printf("),%d,%d]\n",mdfy->ntpt,mdfy->combpt);
      mdfy = mdfy->rnext; }
  }
  if(pluschk>=1 && ringordered==1){
    int k;
    printf("modify ring list (%d) ([mdfy_val,iflip(flips),ntpt,combpt]:\n",
	   nmodify_ring);
    for(k=0; k<flipnum; k++){
      mdfy = modify_init_f[k];
      for(i=0; i<nmodify_ring_f[k]; i++){
	printf("->[%d,%d(", mdfy->val, mdfy->fln);
	for(j=0; j<mdfy->fln; j++){
	  if(j!=0)printf(",");printf("%d",mdfy->flip[j]+1);}
	printf("),%d,%d]\n",mdfy->ntpt,mdfy->combpt);
	mdfy = mdfy->rnext; }
    }
  }
#endif
}

/***** output final results **************************************************/
void output_results(void)
{
  int i, temp, total_weight;
  double endtime;

  endtime=cpu_time();
#if LEVEL == 1
  if(maxmemacc>0){
    temp = sum_memory_access_calc + sum_memory_access_update;
    if((temp % outforevery)!=0){
      printf("%12d",(1+temp/outforevery)*outforevery);
      if(optknown==1)
	printf(" %9.5f   ",(double) 100.0*(optimal-bestgain)/optimal);
      printf(" %10d %8d   %5d %8.3f\n",
	     bestgain,imoves,ntry,cpu_time()-starttime);
    }
  }
  else if(maxmoves==0){
    if((isamples % outforevery)!=0){
      printf("%8d",(1+isamples/outforevery)*outforevery);
      if(optknown==1)
	printf(" %9.5f   ",(double) 100.0*(optimal-bestgain)/optimal);
      printf(" %10d %6d   %5d %8.3f\n",
	     bestgain,imoves,ntry,cpu_time()-starttime);
    }
  }
  else{
    if((imoves % outforevery)!=0){
      printf("%8d",(1+imoves/outforevery)*outforevery);
      if(optknown==1)
	printf(" %9.5f   ",(double) 100.0*(optimal-bestgain)/optimal);
      printf(" %10d %8d   %5d %8.3f\n",
	     bestgain,isamples,ntry,cpu_time()-starttime);
    }
  }
#endif
  if(optfound==1)printf("# optimal solution is found!\n");
  printf("# ");
  if(optknown==1) printf("best err: %9.5f (%%), ",
			 (double) 100.0*(optimal-bestgain)/optimal);
  printf("bestgain:   %7d, number of trials: %5d\n",bestgain,ntry);
  sum_gain = (double) sum_gain / ntry;
  printf("# ");
  if(optknown==1) printf("avr. err: %9.5f (%%), ",
			 (double) 100.0*(optimal-sum_gain)/optimal);
  printf("avr.gain:   %10.2f / trial\n",sum_gain);
  printf("#    total:  %8d samples,  %6d moves, %8.3f secs.\n",
	 isamples,imoves,endtime-starttime);
  printf("#  average: %9.2f samples, %7.2f moves, %8.3f secs. / trial\n",
	 (double)isamples/ntry,(double)imoves/ntry,(endtime-starttime)/ntry);
  printf("# avr. max_mem_used  =%8.2f  max_hash_trace =%7.2f (ideal: %3d)\n",
	 (double)sum_max_nmem_used/ntry, (double)sum_max_hash_trace/ntry,
	 (int)((double)sum_max_nmem_used/ntry)/(1<<hashsize)+1);
  printf("# avr. neighbor size =%8.2f\n",
	 (double) nmodify_ring_sum / imoves);
  printf("# memory access: total = %g (calc: %g, update: %g)\n",
	 sum_memory_access_calc + sum_memory_access_update,
	 sum_memory_access_calc, sum_memory_access_update);
  printf("# memory access: avr.  = %g (calc: %g, update: %g)/move\n",
	 (double)(sum_memory_access_calc + sum_memory_access_update)/imoves,
	 (double)sum_memory_access_calc/imoves,
	 (double)sum_memory_access_update/imoves);
# if EVALBEST==1
  printf("# re-evaluated bestcost = %d\n",evaluate(bestsol,ntrue));
#endif
#if TRUE
  total_weight = 0;
  for(i=0; i<m; i++) total_weight += w[i];
  printf("# weighted sum of unsat clause = %7d\n", total_weight - bestgain);
#endif
  if( LEVEL >= 3 || outbestsol==1 ){
    printf("# best solution:");
    for(i=0;i<org_n;i++)printf(" %1d",bestsol[i]);
    printf("\n");
  }
}

/***** calculate Hansen & Jaumard LB and output ******************************/
void output_HJ_lb(void)
{
  int i, j, k, *hjp, maxr;
  double hjlb, hjlbtemp;

  if((hjp=(int *)malloc((n+1)*sizeof(int)))==NULL) exit(1);
  for(i=0; i<=n; i++) hjp[i]=0;
  maxr=0;
  for(i=0; i<m; i++){
    hjp[nlit[i]]++;
    if(nlit[i] > maxr) maxr = nlit[i];
  }
#if FALSE /* debug */
  for(i=1; i<=maxr; i++) printf("p[%d] = %3d\n", i, hjp[i]);
#endif
  hjlb = 0;
  for(k=1; k<=maxr; k++){
    hjlbtemp = k * m;
    for(i=1; i<k; i++) hjlbtemp -= ( (k - i) * hjp[i] );
    hjlbtemp = (double) hjlbtemp / (k+1);
    /*printf("hjlbtemp = %g\n", hjlbtemp);/**/
    if(hjlbtemp > hjlb) hjlb = hjlbtemp;
  }
  printf("# Hansen & Jaumard LB = %g\n", hjlb);
  free(hjp);
}


/***** free memory space *****************************************************/
void free_memories(void)
{
  int i;
  free(nlit);
  free(w);
  for(i=0;i<m;i++) free(lit[i]);
  free(lit);
  free(nclause[0]);
  for(i=0;i<n;i++) free(clause[0][i]);
  free(clause[0]);
  free(csol);
  free(ntrue);
  if(initrev>0){
    free(csol_rev);
    free(ntrue_rev);}
  if(pluschk==0) free(nbpointer);
#if LEVEL >= 2 || EVALBEST == 1
  free(bestsol);
#endif
  free(free_mems);
  for(i=0; i<nmem_heads; i++) free(mem_heads[i]);
  free(mem_heads);
  free(clchnum);
  free(clchplus);
  free(ntrue_prev);
  free(hashplus);
  free(ref_hashplus);
  if(pluschk==2 || pluschk==4){
    free(fl2_list);
    free(nfl2_list);}
  if(zerochk==1)free(inzeroclause);
}

/***** output current state (level==1) ***************************************/
void output_current_state(void)
{
#if LEVEL == 1
  if(maxmemacc==0){
    if(isamples==0){
      printf("# samples");
      if(optknown==1)printf(" best err(%%) ");
      printf(" best gain  moves  trials     time(secs.)\n");}
    else if((isamples % outforevery)==0){
      printf("%8d",isamples);
      if(optknown==1)
	printf(" %9.5f   ",(double) 100.0*(optimal-bestgain)/optimal);
      printf(" %10d %6d   %5d %8.3f\n",
	     bestgain,imoves,ntry,cpu_time()-starttime);
      if(logoutput==1) outforevery *= 2;
    }
  }
#endif
}

/***** output current state (level==1) ***************************************/
void output_current_state_m(void)
{
#if LEVEL == 1
  if(maxmemacc==0){
    if(imoves==0){
      printf("#  moves ");
      if(optknown==1)printf(" best err(%%) ");
      printf(" best gain  samples  trials     time(secs.)\n");}
    else if((imoves % outforevery)==0){
      printf("%8d",imoves);
      if(optknown==1)
	printf(" %9.5f   ",(double) 100.0*(optimal-bestgain)/optimal);
      printf(" %10d %8d   %5d %8.3f\n",
	     bestgain,isamples,ntry,cpu_time()-starttime);
      if(logoutput==1) outforevery *= 2;
    }
  }
#endif
}

/***** output current state (level==1) ***************************************/
void output_current_state_mem(void)
{
#if LEVEL == 1
  int temp;
  temp = sum_memory_access_calc + sum_memory_access_update;
  if(maxmemacc>0){
    if(temp==0){
      printf("# mem_access ");
      if(optknown==1)printf(" best err(%%) ");
      printf(" best gain    moves  trials     time(secs.)\n");}
    else if((temp % outforevery)==0){
      printf("%12d",temp);
      if(optknown==1)
	printf(" %9.5f   ",(double) 100.0*(optimal-bestgain)/optimal);
      printf(" %10d %8d   %5d %8.3f\n",
	     bestgain,imoves,ntry,cpu_time()-starttime);
      if(logoutput==1) outforevery *= 2;
    }
  }
#endif
}

/***** update bestgain and bestsol *******************************************/
void update_bestsol(void)
{
  int i;
#if LEVEL >= 2 || EVALBEST == 1
  for(i=0;i<n;i++) bestsol[i] = csol[i];
#endif
}

/***** calculate flip gain ***************************************************/
int flipgain(int iflip, int *ii, int *ii1)
{
  struct modify *mdfy;
  char space_found, space_empty;
  int i, j, k, flgain, hashvalue;

  flgain=0;
  for(i=0;i<=iflip;i++)ii1[i]=iflip+1;
  for(i=1;i<=iflip;i++){
    for(j=0;j<i;j++)ii1[j]=j;
    while(ii1[i-1]<iflip){
      sum_memory_access_calc ++; /* count the memory access for flip gain */
      output_current_state_mem();
      /* calculate hash value */
      hashvalue = 0;
      for(j=0; j<i; j++)
#if HASHWOOF==1
	hashvalue=(((hashvalue*(n&hashmask))&hashmask)+ii[ii1[j]]) & hashmask;
#else
	hashvalue=((hashvalue*n)+ii[ii1[j]]) & hashmask;
#endif
      /* check the existance in the hash */
      if(hash[hashvalue]==NULL){space_found = 0;}
      else{
	mdfy = hash[hashvalue];
	space_empty = 0;
	do{
	  space_found = 1;
	  if(mdfy->fln != i){
	    space_found = 0;
	    if(mdfy->hnext!=NULL){mdfy=mdfy->hnext;}
	    else space_empty = 1;}
	  else for(j=0; j<i; j++) if(mdfy->flip[j]!=ii[ii1[j]]){
	    space_found = 0;
	    if(mdfy->hnext!=NULL){mdfy=mdfy->hnext;}
	    else space_empty = 1;
	    break;}
	}while((space_found | space_empty)==0);
      }
      /**** add to the flgain ****/
      if(space_found==1) flgain += mdfy->val;
      /**** generate the next candidate ****/
      j=0;
      while(ii1[j]==ii1[j+1]-1)j++;
      ii1[j]++;
      for(k=0;k<j;k++)ii1[k]=k;
    }
  }
#if FALSE /**** check the correctness ****/
  /*printf("current ntrue:");
  for(i=0;i<m;i++)printf("%3d",ntrue[i]);printf("\n");*/
  for(i=0;i<iflip;i++) csol[ii[i]] = 1 - csol[ii[i]];
  printf("iflip: %d,  true gain: %7d,  calculated gain: %7d, flips:",
	 iflip, evaluate(csol,ntrue), cgain+flgain);
  for(i=0;i<iflip;i++)printf("%3d",ii[i]+1);printf("\n");
  for(i=0;i<iflip;i++) csol[ii[i]] = 1 - csol[ii[i]];
  evaluate(csol,ntrue);
#endif
  return(flgain);
}

/***** decode nbpointer ******************************************************/
void decode_nbpointer(int pt, int *iflip, int *ii)
{
  int i, temp, fl;
  fl = nbpointer[pt] & MASK;
  temp = 1;
  for(i=0;i<fl;i++){
    ii[i] = (nbpointer[pt] >> BITSFORFLIP)/temp % n;
    temp *= n;}
  *iflip = fl;
#if FALSE
  printf("%5d %3d:", pt+1, *iflip);
  for(i=0;i<*iflip;i++) printf("%3d",ii[i]+1);
  printf("\n");
#endif
}

/***** check if the nbsize is too large or not *******************************/
void check_nbsize(void)
{
  int i; unsigned long sum, temp;
  temp=1; sum=0;
  for(i=0;i<flipnum;i++){
    if(temp <= (INT_MAX-sum)/(n-1)) sum += ((n-1)*temp);
    else{
      printf("WARNING: neighbor size too large\n");
      exit(1);}
    if(temp <= INT_MAX/n) temp *= n;
    else{
      printf("WARNING: neighbor size too large\n");
      exit(1);}
  }
  if(sum > (INT_MAX >> (BITSFORFLIP-1))){
    printf("WARNING: neighbor size too large\n");
    exit(1);}
}

/***** malloc with error check ***********************************************/
void *malloc_e( size_t size ) {
  void *s;
  if ( (s=malloc(size)) == NULL ) {
    fprintf( stderr, "malloc : Not enough memory.\n" );
    exit( EXIT_FAILURE );
  }
  return s;
}

/***** randomize nbpointer ***************************************************/
void randomize_nbpointer(void)
{
  int i, j;
  unsigned long temp;

  for(i=0;i<nbsize;i++){
    j=rand_from(i,nbsize-1);
    temp=nbpointer[i];
    nbpointer[i]=nbpointer[j];
    nbpointer[j]=temp;}
}

/***** initialize nbpointer *************************************************/
void initialize_nbpointer(void)
{
  int i, j, k, nbsize, *ii, temp;
  unsigned long encoded_cand, *nbp_temp;

  /*nbsize_tmp=0;/**/
  nbp_temp=nbpointer;
  if((ii=(int *)malloc((flipnum+1)*sizeof(int)))==NULL) exit(1);
  for(i=0;i<=flipnum;i++)ii[i]=n+1;
  for(i=1;i<=flipnum;i++){
    for(j=0;j<i;j++)ii[j]=j;
    while(ii[i-1]<n){
      /**** store the neighbor candidate ****/
      encoded_cand = i;
      for(j=0,temp=1;j<i;j++){
	encoded_cand += ((ii[j]*temp) << BITSFORFLIP);
	temp *= n;}
      *nbp_temp++ = encoded_cand;
      /*nbsize_tmp++;/**/
      /**** generate the next candidate ****/
      j=0;
      while(ii[j]==ii[j+1]-1)j++;
      ii[j]++;
      for(k=0;k<j;k++)ii[k]=k;
    }
  }
  free(ii);
#if FALSE /* the above 'nbsize_tmp' should be calculated to make this TRUE */
  printf("real nbsize = %d\n", nbsize_tmp);
  printf("index flip variables\n");
  for(i=0;i<nbsize_tmp;i++){
    k =  nbpointer[i] & MASK;
    printf("%5d %3d:", i+1, k);
    temp = 1;
    for(j=0;j<k;j++){
      printf("%3d",((nbpointer[i] >> BITSFORFLIP)/temp) % n);
      temp *= n;}
    printf("\n");}
#endif
}

/***** calculate n choose k *************************************************/
int i_choose_j(int i, int j)
{
#if FALSE
  int k, ans;
  ans=1;
  for(k=0;k<j;k++) ans *= (i-k);
  for(k=j;k>1;k--) ans /= k;
  return(ans);
#else
  int temp1;
  unsigned long temp;
  if(j<=0) return(1);
  else{
    temp=i_choose_j(i,j-1);
    temp1=i-j+1;
    if(temp1>0)if(temp>INT_MAX/temp1){ /** check overflow **/
      printf("nbsize cannot be calculated correctly!\n");
      exit(1);}
    return (temp*temp1/j);
  }
#endif
}

/***** subroutine for generating random sequence of numbers ******************/
int rand_from(int min, int max)
{
  x_n *= 2100005341;
  return((int) ((x_n / 4294967296.0) * (max-min+1)) + min);
}

/***** evaluate the gain of 'sol' from scratch *******************************/
int evaluate(char *sol, int *nt)
{
  int i, j, nclause_i, sum;
  char truelit;

  for(i=0;i<m;i++) nt[i]=0;
  for(i=0;i<n;i++){
    truelit=1-sol[i];
    nclause_i=nclause[truelit][i];
    for(j=0;j<nclause_i;j++) nt[clause[truelit][i][j]]++;}
  sum=0;
  for(i=0;i<m;i++)if(nt[i]>0)sum+=w[i];
  return(sum);
}

/***** generate a solution randomly ******************************************/
void random_init(void)
{
  int i;

  for(i=0;i<n;i++)csol[i]=rand_from(0,1);
  cgain=evaluate(csol,ntrue);
  isamples++;
  if(initrev>0){ /* consider the negated solution */
    for(i=0;i<n;i++)csol_rev[i]=1-csol[i];
    cgain_rev=evaluate(csol_rev,ntrue_rev);
    isamples++;
    if(initrev==1){ /* choose better solution for init sol */
      if(cgain_rev>cgain) swap_csol_and_csol_rev();}
  }
#if LEVEL >= 2  /**** beginning of DEBUG ****/
  printf("init sol[%3d]: ",cgain);
  if(LEVEL>=3)for(i=0;i<n;i++)printf("%1d ", csol[i]);
  if(initrev>0){
    printf("\n reverse[%3d]: ",cgain_rev);
    if(LEVEL>=3)for(i=0;i<n;i++)printf("%1d ", csol_rev[i]);
  }
  printf("\n");
#endif          /**** end of DEBUG ****/
}

/***** swap csol and csol_rev ************************************************/
void swap_csol_and_csol_rev(void)
{
  int temp, *ntemp;
  char *ptemp;
  ptemp = csol;   csol  = csol_rev;   csol_rev  = ptemp;
  ntemp = ntrue;  ntrue = ntrue_rev;  ntrue_rev = ntemp;
  temp  = cgain;  cgain = cgain_rev;  cgain_rev = temp;
}

/***** remove reduncancy of the input data (unnecessary lit & clause) ********/
void rm_redundancy(void)
{
  int i, j, k, nlit_i, var_ij, *marked_var, nmarked_var,
      nredundant;
  char *mark, sign_lit_ij, *redundant;

  /**** initialize memory ****/
  mark = (char *) malloc_e( (n+1) * sizeof(char) );
  marked_var = (int *) malloc_e( n * sizeof(int) );
  redundant = (char *) malloc_e( n * sizeof(char) );
  for(i=0; i<=n; i++) mark[i]=0;
  org_n = n;

  /**** check routine ****/
  for(i=0; i<m; i++){
    nmarked_var = nredundant = 0;
    nlit_i = nlit[i];
    for(j=0; j<nlit_i; j++){
      var_ij = lit[i][j];
      if(var_ij > 0){ sign_lit_ij = 1; }
      else{ var_ij = - var_ij; sign_lit_ij = -1; }
      if(mark[var_ij]==0){
	mark[var_ij] = sign_lit_ij;
	marked_var[nmarked_var] = var_ij;
	nmarked_var++;
      }
      else if(mark[var_ij] == sign_lit_ij){ /* same lit with same sign */
	if(nredundant==0)for(k=0; k<nlit_i; k++)redundant[k]=0;
	redundant[j] = 1;
	nredundant++;
      }
      else{
	n = org_n + 1;
	nlit[i] = 1;
	lit[i][0] = n;
	nredundant = 0;
	break;
      }
    }
    if(nredundant>0){
      k = 0;
      for(j=0; j<nlit_i; j++)if(redundant[j]==0){
	lit[i][k] = lit[i][j];
	k++;
      }
      nlit[i] = k;
    }
    for(k=0; k<nmarked_var; k++)mark[marked_var[k]]=0;
  }

  /**** free memory ****/
  free(mark);
  free(marked_var);
  free(redundant);

#if FALSE /* debug (output the reduced instance) */
  printf("%d %d\n", n, m);
  for(i=0; i<m; i++){
    printf("%2d %4d", nlit[i], w[i]);
    for(j=0; j<nlit[i]; j++)printf(" %2d", lit[i][j]);
    printf("\n");
  }
#endif
}

/***** read problem data *****************************************************/
void read_data(void)
{
  char  file[15], *mark, tempstr[15];
  int   i, j, k, temp, *lit_temp, nlit_i, **frontier[2];
  FILE  *fp=stdin;

  /**** input file name ****/
#if FALSE
    printf("Input Instance File-name : ");
    scanf("%s",file); printf("\n");
    if(NULL==(fp=fopen(file, "r"))){
      printf("\n Cannot Open File : %s\n\n",file);
      exit(1);}
#endif
  fscanf(fp,"%s",tempstr);
  if(strcmp(tempstr,"opt")==0)optknown=1;
  else optknown=0;
  if(optknown==1){
    fscanf(fp,"%d",&optimal);  /* read the optimal value */
#if PRINT_INPUT==1
    printf("%3d ",optimal);
#endif
    fscanf(fp,"%d",&n);        /* read the number of variables */
  }
  else{
    optimal=0;
    n=atoi(tempstr);
  }
  fscanf(fp,"%d",&m);          /* read the number of clauses */
#if PRINT_INPUT==1
  printf("%3d ",n);
  printf("%3d \n",m);
#endif
  if((nlit=(int *)malloc(m*sizeof(int)))==NULL) exit(1);
  if((w=(int *)malloc(m*sizeof(int)))==NULL) exit(1);
  if((lit=(int **)malloc(m*sizeof(int *)))==NULL) exit(1);
  for(i=0;i<m;i++){
    fscanf(fp,"%d",nlit+i);    /* read #variables in clause i */
    fscanf(fp,"%d",w+i);       /* read the weight of clause i */
    if((*(lit+i)=lit_temp=(int *)malloc((*(nlit+i))*sizeof(int)))==NULL) 
      exit(1);
    nlit_i=nlit[i];
    for(j=0;j<nlit_i;j++){
      fscanf(fp,"%d",lit_temp+j);} /* read the literals of clause i */
  }
/*  fclose(fp);*/
#if PRINT_INPUT==1
  printf("clause #lit weight literals\n");
  for(i=0;i<m;i++){
    nlit_i=nlit[i];
    printf("%4d  [  %2d, %3d]:",i+1,nlit_i,w[i]);
/*    for(j=0;j<nlit_i;j++)printf("%3d ",*(*(lit+i)+j));*/
/*    for(j=0;j<nlit_i;j++)printf("%3d ",*(lit[i]+j));*/
    for(j=0;j<nlit_i;j++)printf("%3d ",lit[i][j]);
    printf("\n");
  }
#endif
#if REDUNDANTCHK==1  /***** check redundancy *****/
  for(i=0;i<m;i++){
    nlit_i=nlit[i];
    if(nlit_i==1){printf("only one variable!\n");}
    else{
      for(j=0;j<nlit_i-1;j++){
	for(k=j+1;k<nlit_i;k++){
	  if(lit[i][j]==lit[i][k]){printf("same variable!\n");}
	  if(lit[i][j]==-lit[i][k]){
	    printf("%d is unnecessary clause!\n", i+1);}
	}}}
  }
  if((mark=(char *)malloc((2*n+1)*sizeof(char)))==NULL) exit(1);
  for(i=0;i<m-1;i++){
    for(j=0;j<=2*n;j++)mark[j]=0;
    for(j=0,nlit_i=nlit[i];j<nlit_i;j++){
      if(lit[i][j]<0)mark[n-lit[i][j]]=1;
      else mark[lit[i][j]]=1;}
    for(j=i+1;j<m;j++)if(j!=i && nlit[i]==nlit[j]){
      k=0;
      while((lit[j][k]<0 && mark[n-lit[j][k]]==1)
	    || (lit[j][k]>0 && mark[lit[j][k]]==1)){
	if(k<nlit[j]-1) k++;
	else{
	  printf("same clauses!\n");
	  break;}
      }
    }
  }
  free(mark);
#endif
  rm_redundancy();/**/
  /**** count the number of pos/neg clauses for each variable ****/
  if((nclause[0]=(int *)malloc(2*n*sizeof(int)))==NULL) exit(1);
  nclause[1]=nclause[0]+n;
  for(i=0;i<n;i++)nclause[0][i]=nclause[1][i]=0;
  for(i=0;i<m;i++){
    nlit_i=nlit[i];
    for(j=0;j<nlit_i;j++){
      temp=lit[i][j];
      if(temp<0) nclause[1][-temp-1]++;
      else nclause[0][temp-1]++;}
  }
  /**** make the clause list for each literal ****/
  if((clause[0]=(int **)malloc(2*n*sizeof(int *)))==NULL) exit(1);
  clause[1]=clause[0]+n;
  for(i=0;i<n;i++){
    if((clause[0][i]=(int *)malloc((nclause[0][i]+nclause[1][i])*sizeof(int)))
       ==NULL) exit(1);
    clause[1][i]=clause[0][i]+nclause[0][i];
  }
  if((frontier[0]=(int **)malloc(2*n*sizeof(int *)))==NULL) exit(1);
  frontier[1]=frontier[0]+n;
  for(i=0;i<n;i++){
    frontier[0][i]=clause[0][i];
    frontier[1][i]=clause[1][i];}
  for(i=0;i<m;i++){
    nlit_i=nlit[i];
    for(j=0;j<nlit_i;j++){
      temp=lit[i][j];
      if(temp<0) *frontier[1][-temp-1]++ = i;
      else       *frontier[0][temp-1]++  = i;}}
  free(frontier[0]);
#if PRINT_INPUT==1
  printf("val npos/nneg  clauses");
  for(i=0;i<n;i++){
    printf("\n%3d npos=%3d:",i+1,nclause[0][i]);
    for(j=0;j<nclause[0][i];j++)printf("%3d ",clause[0][i][j]+1);
    printf("\n    nneg=%3d:",nclause[1][i]);
    for(j=0;j<nclause[1][i];j++)printf("%3d ",clause[1][i][j]+1);}
  printf("\n");
#endif
#if TRUE /* compute the max appearance of variables and max clause size */
  max_var_appear=max_clause_size=avr_var_appear=avr_clause_size=0;
  for(i=0; i<m; i++){
    avr_clause_size += nlit[i];
    if(nlit[i]>max_clause_size)max_clause_size=nlit[i];
  }
  for(i=0; i<n; i++){
    avr_var_appear += (nclause[0][i]+nclause[1][i]);
    if(nclause[0][i]+nclause[1][i]>max_var_appear)
      max_var_appear = nclause[0][i]+nclause[1][i];
  }
  avr_clause_size /= m;
  avr_var_appear  /= n;
#endif
}

/***** initialize memories ***************************************************/
void init_mems(void)
{
  int i;

  if((csol=(char *)malloc(n*sizeof(char)))==NULL) exit(1);
  if((ntrue=(int *)malloc(m*sizeof(int)))==NULL) exit(1);
  if(initrev>0){
    if((csol_rev=(char *)malloc(n*sizeof(char)))==NULL) exit(1);
    if((ntrue_rev=(int *)malloc(m*sizeof(int)))==NULL) exit(1);}
#if LEVEL >= 2 || EVALBEST == 1
  if((bestsol=(char *)malloc(n*sizeof(char)))==NULL) exit(1);
#endif
  hashmask = (1 << hashsize) - 1;
  if((hash=(struct modify **)malloc((hashmask+1)*sizeof(struct modify *)))
     ==NULL) exit(1);
  for(i=0; i<hashmask+1; i++) hash[i] = NULL;
  if((hashplus=(int *)malloc((hashmask+1)*sizeof(int)))==NULL) exit(1);
  if((ref_hashplus=(int *)malloc((hashmask+1)*sizeof(int)))==NULL) exit(1);
  nhashplus = 0;
  if((mem_heads=(struct modify **)malloc(mallocunit*sizeof(struct modify *)))
     ==NULL) exit(1);
  mem_heads_size = mallocunit;
  nmem_heads = 0;
  if((free_mems=(struct modify **)malloc(mallocunit*sizeof(struct modify *)))
     ==NULL) exit(1);
  free_mems_size = mallocunit;
  nfree_mems = 0;
  nmem_used = max_nmem_used = max_hash_trace = 0;
  sum_max_nmem_used = sum_max_hash_trace = 0;
  /* list of changed clause */
  if((clchnum=(int *)malloc(m*sizeof(int)))==NULL) exit(1);
  for(i=0; i<m; i++) clchnum[i] = 0;
  if((clchplus=(int *)malloc(m*sizeof(int)))==NULL) exit(1);
  if((ntrue_prev=(int *)malloc(m*sizeof(int)))==NULL) exit(1);
  /* fl2_list */
  if(pluschk==2 || pluschk==4){
    if((nfl2_list=(int *)malloc(n*sizeof(int)))==NULL) exit(1);
    if((fl2_list=(struct modify **)malloc(n*sizeof(struct modify *)))==NULL)
      exit(1);
  }
  /* zerochk */
  if(zerochk==1)if((inzeroclause=(int *)malloc(n*sizeof(int)))==NULL) exit(1);
}

/***** make neighbor order list **********************************************/
void make_nb_order_list(void)
{
  int i;
  nbsize=0;
  for(i=0;i<flipnum;i++) nbsize += i_choose_j(n,i+1);
  check_nbsize();
  if((nbpointer=(unsigned long *)malloc(nbsize*sizeof(unsigned long)))==NULL)
    exit(1);
  initialize_nbpointer();
}

/***** initialize modify list according to csol ******************************/
void initialize_modify(void)
{
  int i, j, *variables, nlit_i, lit_i_j, npos, nneg;

  nmodify_ring = 0;
  if(pluschk==2 || pluschk==4)for(i=0; i<n; i++) nfl2_list[i]=0;
  if(ringordered==1)for(i=0; i<flipnum; i++) nmodify_ring_f[i]=0;
  if((variables=(int *)malloc(n*sizeof(int)))==NULL) exit(1);
  for(i=0; i<m; i++)if(ntrue[i]<=flipnum){
    nlit_i = nlit[i];
    npos = nneg = 0;
    for(j=0; j<nlit_i; j++){
      lit_i_j = lit[i][j];
      if(lit_i_j>0){
	lit_i_j--;
	if(csol[lit_i_j]==1){
	  variables[npos] = lit_i_j;
	  npos++;}
	else{
	  variables[flipnum+nneg] = lit_i_j;
	  nneg++;}
      }
      else{
	lit_i_j = -lit_i_j-1;
	if(csol[lit_i_j]==0){
	  variables[npos] = lit_i_j;
	  npos++;}
	else{
	  variables[flipnum+nneg] = lit_i_j;
	  nneg++;}
      }
    }
    add_to_modify(npos,nneg,variables,variables+flipnum,w[i],1);
  }
#if FALSE /* display initial modify ring */
  printf("# number of plus cells = %5d\n",nmodify_ring);/**/
  if(TRUE){
    int iflip, *ii;
    struct modify *modify_temp;
    modify_temp = modify_init;
    printf("initial modify ring:");
    for(i=0; i<nmodify_ring; i++){
      iflip = modify_temp->fln;
      ii = (modify_temp->flip);
      printf(" ->[%d(", iflip);
      for(j=0; j<iflip; j++)printf(" %d", ii[j]+1);
      printf(")]");
      modify_temp = modify_temp->rnext;
    }
    printf("\n");
  }
#endif
  free(variables);
}

/***** add to modify lists ***************************************************/
void add_to_modify(int npos, int nneg, int *pos, int *neg, int w_i,
		   char addsign)
{
  int i, j, k, uu, *ii, *ii1, temp, hashvalue, mdfy_val, mdfy_val_prev,
      hashtrace, temp1;
  char iflip, swap_occur, hash_empty, space_found, space_empty;
  struct modify *mdfy, *mdfynew, *mdfytemp, *mdfyprev;

  uu = flipnum - npos;
  if(uu > nneg) uu = nneg;
  if((ii=(int *)malloc((uu+flipnum+1)*sizeof(int)))==NULL) exit(1);
  ii1 = ii+flipnum;
  for(i=0; i<=uu; i++) ii1[i] = nneg+1;
  for(i=0; i<=uu; i++){
    iflip = i + npos;
    for(j=0; j<i; j++) ii1[j] = j;
    for(;;){
      if(npos==0 && i==0) break;

      /******** the beggining of the body ********/
      for(k=0; k<npos; k++) ii[k] = pos[k];
      for(k=0; k<i; k++) ii[npos+k] = neg[ii1[k]];
      /* sort ii[] (increasing order) using bubble sort */
      for(k=0; k<iflip; k++){
	swap_occur = 0;
	for(j=iflip-1; j>k; j--)if(ii[j-1]>ii[j]){
	  swap_occur = 1;
	  temp    = ii[j-1];
	  ii[j-1] = ii[j];
	  ii[j]   = temp;}
	if(swap_occur==0) break;
      }
      /*printf("ii:");for(k=0;k<iflip;k++)printf("%3d",ii[k]);printf("\n");/**/
      mdfy_val = addsign * (((i&1)<<1)-1)*w_i; /* calculate the modify value */
      add_to_hash(iflip, ii, mdfy_val, &hashvalue, &mdfy_val_prev,
		  &mdfy, &mdfyprev);
      /*printf("mf:");for(k=0;k<iflip;k++)
	printf("%3d",mdfy->flip[k]);printf("\n");/**/
      if(i==1) (mdfy->ntpt) += addsign;
      if(mdfy->fln == 2 && mdfy_val_prev <= 0 && mdfy->val > 0 &&
	 (pluschk == 2 || pluschk == 4))
	fl2_add(mdfy);
      if(mdfy->fln == 2 && mdfy_val_prev > 0 && mdfy->val <= 0 &&
	 (pluschk == 2 || pluschk == 4))
	fl2_delete(mdfy);
      if(rmzeromem == 1 && (pluschk == 2 || pluschk == 4) && mdfy->fln == 2 &&
	 ((mdfy_val_prev <= 0 && mdfy->val > 0) ||
	  (mdfy_val_prev > 0 && mdfy->val <= 0)) )
	add_to_hash(iflip, ii, 0, &hashvalue, &mdfy_val_prev,&mdfy, &mdfyprev);
      /*printf("mdfy->ntpt = %2d\n",mdfy->ntpt);/**/
      /*if(mdfyprev!=NULL)printf("next to mdfyprev: %d\n",mdfyprev->hnext);/**/
      add_to_ring(hashvalue, mdfy, mdfyprev);
      /*if(mdfyprev!=NULL)printf("next to mdfyprev: %d\n",mdfyprev->hnext);/**/
      /******** the end of the body **************/

      if(i==0) break;
      j=0;
      while(ii1[j]==ii1[j+1]-1)j++;
      ii1[j]++;
      for(k=0; k<j; k++) ii1[k]=k;
      if(ii1[i-1]>=nneg) break;
    }
  }
  free(ii);
}

/***** add a cell to the hash ************************************************/
void add_to_hash(char iflip, int *ii, int mdfy_val, int *hashvl, int *mdfy_vp,
		 struct modify **mdfyp, struct modify **mdfyprevp)
{
  int j, hashvalue, mdfy_val_prev, hashtrace;
  char hash_empty, space_found, space_empty;
  struct modify *mdfy, *mdfyprev, *mdfynew;

  sum_memory_access_update ++; /*count the memory access for update*/
  output_current_state_mem();
  /* calculate hash value */
  hashvalue = 0;
  for(j=0; j<iflip; j++)
#if HASHWOOF==1
    hashvalue=(((hashvalue*(n&hashmask))&hashmask)+ii[j]) & hashmask;
#else
  hashvalue=((hashvalue*n)+ii[j]) & hashmask;
#endif
  /*printf("hash value = %5d\n",hashvalue);/**/
  /* check the existance in the hash */
  hashtrace = 1;
  mdfyprev = NULL;
  if(hash[hashvalue]==NULL){
    hash_empty = 1;	space_found = 0;}
  else{
    hash_empty = 0;
    mdfy = hash[hashvalue];
    space_empty = 0;
    do{
      space_found = 1;
      if(mdfy->fln != iflip){
	space_found = 0;
	if(mdfy->hnext!=NULL){mdfyprev=mdfy;mdfy=mdfy->hnext;hashtrace++;}
	else space_empty = 1;}
      else for(j=0; j<iflip; j++) if(mdfy->flip[j]!=ii[j]){
	space_found = 0;
	if(mdfy->hnext!=NULL){mdfyprev=mdfy;mdfy=mdfy->hnext;hashtrace++;}
	else space_empty = 1;
	break;}
    }while((space_found | space_empty)==0);
  }
  if(space_found == 0){ /* create new memory space */
    if(nfree_mems==0){
      if(nmem_heads >= mem_heads_size){
	mem_heads_size += mallocunit;
	if((mem_heads=(struct modify **)
	    realloc(mem_heads,
		    (mem_heads_size)*sizeof(struct modify *)))
	   ==NULL) exit(1);}
      if((mem_heads[nmem_heads]=(struct modify *)
	  malloc(mallocunit*sizeof(struct modify)))==NULL) exit(1);
      for(j=0; j<mallocunit; j++) free_mems[j] = mem_heads[nmem_heads]+j;
      nmem_heads++;
      nfree_mems=mallocunit;
#if FALSE /* debug */
      printf("free_mems created:");
      for(j=0; j<mallocunit; j++) printf(" %d",free_mems[j]);printf("\n");
#endif
    }
    nfree_mems--;
    mdfynew = free_mems[nfree_mems];
    /* fill the values into the new memory space */
    mdfynew->fln = iflip;
    for(j=0; j<iflip; j++) mdfynew->flip[j] = ii[j];
    mdfynew->val = mdfy_val;
    mdfynew->hnext = NULL;
    mdfynew->ntpt = 0;
    mdfynew->inring = 0;
    mdfynew->combpt = 0;
    if(hash_empty==1){
      hash[hashvalue] = mdfynew;
      hashplus[nhashplus]=hashvalue;
      ref_hashplus[hashvalue]=nhashplus;
      nhashplus++;}
    else mdfy->hnext = mdfynew; /*if(space_empty==1)*/
    mdfy_val_prev = 0;
    mdfy = mdfynew;
    nmem_used++;
    if(nmem_used > max_nmem_used) max_nmem_used = nmem_used;
    if(hashtrace > max_hash_trace) max_hash_trace = hashtrace;
  }
  else{ /* modify the modify value */
    mdfy_val_prev = mdfy->val;
    mdfy->val += mdfy_val;
  }

  *hashvl = hashvalue;
  *mdfy_vp = mdfy_val_prev;
  *mdfyp = mdfy;
  *mdfyprevp = mdfyprev;
#if FALSE /* debug */
  if(space_found == 0) printf("create[%d]:", mdfy);
  else printf("modify[%d]:", mdfy);
  printf(" val: %3d->%3d, combpt:%2d, ntpt:%2d, ",
	 mdfy_val_prev,mdfy->val,mdfy->combpt,mdfy->ntpt);
  printf("%d flips(",mdfy->fln);
  for(j=0; j<mdfy->fln; j++)printf("%3d",mdfy->flip[j]+1);
  printf(")");
  if(mdfyprev != NULL){
    printf(" prev cell[%d]: %d flips(", mdfyprev, mdfyprev->fln);
    for(j=0; j<mdfyprev->fln; j++)printf("%3d",mdfyprev->flip[j]+1);
    printf(")");}
  printf("\n");
  mdfynew = hash[hashvalue];
  for(;;){
    printf("->[%d]%d flips(", mdfynew, mdfynew->fln);
    for(j=0; j<mdfynew->fln; j++)printf("%3d",mdfynew->flip[j]+1);
    printf(")");
    if(mdfynew->hnext == NULL) break;
    else mdfynew = mdfynew->hnext;
  }printf("\n");
#endif
}

/***** add a cell to the candidate ring **************************************/
void add_to_ring(int hashvalue, struct modify *mdfy, struct modify *mdfyprev)
{
  int j, temp, temp1;
  struct modify *mdfytemp;

  if(pluschk>=1){ /* modify the ring list */
    if(mdfy->inring == 0 &&                      /* add to the ring */
       ((mdfy->val > 0 && 
	 (pluschk == 1 || pluschk == 2 ||
	  ((pluschk == 3 || pluschk == 4) && mdfy->ntpt > 0))) ||
	mdfy->combpt > 0) ){
      mdfy->inring = 1;
      if(ringordered==0){
	if(nmodify_ring==0){
	  modify_init = mdfy;
	  mdfy->rprev = mdfy->rnext = mdfy;}
	else{
	  mdfytemp = modify_init->rprev;
	  mdfytemp->rnext = mdfy;
	  modify_init->rprev = mdfy;
	  mdfy->rprev = mdfytemp;
	  mdfy->rnext = modify_init;}
      }
      else{/* ringoerdered = 1 */
	j = mdfy->fln - 1;
	if(nmodify_ring_f[j]==0){
	  modify_init_f[j] = mdfy;
	  mdfy->rprev = mdfy->rnext = mdfy;}
	else{
	  mdfytemp = modify_init_f[j]->rprev;
	  mdfytemp->rnext = mdfy;
	  modify_init_f[j]->rprev = mdfy;
	  mdfy->rprev = mdfytemp;
	  mdfy->rnext = modify_init_f[j];}
	nmodify_ring_f[j]++;
      }
      /*printf("add\n");/**/
      nmodify_ring++;
    }
    else if(mdfy->inring == 1 &&       /* delete from the ring */
	    (mdfy->val <= 0 ||
	     ((pluschk == 3 || pluschk == 4) && mdfy->ntpt <= 0)) &&
	    mdfy->combpt <= 0 ){
      mdfy->inring = 0;
      if(ringordered==0){
	if(nmodify_ring==1){
	  modify_init = NULL;}
	else{
	  if(modify_init==mdfy) modify_init = mdfy->rnext;
	  (mdfy->rprev)->rnext = mdfy->rnext;
	  (mdfy->rnext)->rprev = mdfy->rprev; }
      }
      else{/* ringordered = 1 */
	j = mdfy->fln - 1;
	if(nmodify_ring_f[j]==1){
	  modify_init_f[j] = NULL;}
	else{
	  if(modify_init_f[j]==mdfy) modify_init_f[j] = mdfy->rnext;
	  (mdfy->rprev)->rnext = mdfy->rnext;
	  (mdfy->rnext)->rprev = mdfy->rprev; }
	nmodify_ring_f[j]--;
      }
      /*printf("delete\n");/**/
      nmodify_ring--;
    }
  }
  if(mdfy->combpt < 0 || mdfy->ntpt < 0){
    printf("error! iflip: %d combpt: %d ntpt: %d\n",
	   mdfy->fln,mdfy->combpt,mdfy->ntpt);    exit(1);}/**/
  if(rmzeromem == 1){                              /*remove unnecessary mems */
    if(mdfy->val == 0 && mdfy->combpt <= 0 && mdfy->ntpt <= 0
       /*&& mdfy->fln !=2*/){
#if FALSE
      printf("rm cell[%d]: %d flips(", mdfy, mdfy->fln);
      for(j=0; j<mdfy->fln; j++) printf("%3d", mdfy->flip[j]+1);
      printf(")");
      if(mdfyprev != NULL){printf(", prev cell[%d]: %d flips(", mdfyprev, mdfyprev->fln);
      for(j=0; j<mdfyprev->fln; j++) printf("%3d", mdfyprev->flip[j]+1);
      printf(")");}
      if(mdfy->hnext!=NULL){printf(", next cell: %d flips(", mdfy->hnext->fln);
      for(j=0; j<mdfy->hnext->fln; j++) printf("%3d", mdfy->hnext->flip[j]+1);
      printf(")");}
      printf("\n");
      printf("next_to_prev_cell = ");
      if(mdfyprev==NULL){printf("%d\n", hash[hashvalue]);}
      else{printf("%d\n", mdfyprev->hnext);}
#endif
      if(mdfyprev==NULL){
	hash[hashvalue] = mdfy->hnext;
	if(hash[hashvalue] == NULL){
	  nhashplus--;
	  temp  = hashplus[nhashplus];
	  temp1 = ref_hashplus[hashvalue];
	  hashplus[temp1] = temp;
	  ref_hashplus[temp] = temp1;
	}
      }
      else mdfyprev->hnext = mdfy->hnext;
      if(nfree_mems >= free_mems_size){
	free_mems_size += mallocunit;
	if((free_mems=(struct modify **)
	    realloc(free_mems,free_mems_size*sizeof(struct modify *)))
	   ==NULL) exit(1);
      }
      free_mems[nfree_mems] = mdfy;
      nfree_mems++;
      nmem_used--;
#if FALSE /* debug */
      printf("next_to_prev_cell = ");
      if(mdfyprev==NULL){printf("%d\n", hash[hashvalue]);}
      else{printf("%d\n", mdfyprev->hnext);}
      printf("free_mems created: %d (nfree_mems = %d)\n",
	     free_mems[nfree_mems-1], nfree_mems);
      for(j=nfree_mems-100; j<nfree_mems; j++){
	if(j>0 && j%10==0)printf("\n");
	printf(" %d",free_mems[j]);
      }
      printf("\n");
#endif
    }
  }
}

/***** add necessary cells to the candidate ring (pluschk=2,4) ***************/
void fl2_add(struct modify *mdfy)
{
  int i, j, ix, i0, i1, i2, irev, ii[3], nfl2_list_ix, hashvalue, mdfy_vp;
  struct modify *mdfy2, *mdfy_n, *mdfy_p, *mdfy3, *mdfy3prev;

  i0 = mdfy->flip[0];
  i1 = mdfy->flip[1];
  if(i0>i1){printf("order of flips are wrong!\n");exit(1);}/**/
  for(i=0; i<2; i++){
    ix = mdfy->flip[i];
#if FALSE /* debug */
    nfl2_list_ix = nfl2_list[ix];
    mdfy2 = fl2_list[ix];
    printf("   add(%3d%3d): fl2_list[%3d]",i0+1,i1+1,ix+1);
    for(j=0; j<nfl2_list_ix; j++){
      if(mdfy2->flip[0] == ix) irev = 1;
      else irev = 0;
      printf("->%3d", mdfy2->flip[irev]+1);
      mdfy2 = mdfy2->fl2_next[irev];
    }printf("\n");
#endif
    irev = 1 - i;
    if(nfl2_list[ix] <= 0){
      if(nfl2_list[ix] < 0){printf("error around fl2_list!\n");exit(1);}/**/
      fl2_list[ix] = mdfy;
      mdfy->fl2_next[irev] = mdfy->fl2_prev[irev] = mdfy;
    }
    else{
      /* add to fl2_list */
      mdfy_n = fl2_list[ix];
      if(mdfy_n->flip[0] == ix) irev = 1;
      else irev = 0;
      mdfy_p = mdfy_n->fl2_prev[irev];
      mdfy_n->fl2_prev[irev] = mdfy;
      irev = 1 - i;
      mdfy->fl2_next[irev] = mdfy_n;
      mdfy->fl2_prev[irev] = mdfy_p;
      if(mdfy_p->flip[0] == ix) irev = 1;
      else irev = 0;
      mdfy_p->fl2_next[irev] = mdfy;
      /* end of 'add to fl2_list' */
#if FALSE /* debug */
      nfl2_list_ix = nfl2_list[ix];
      mdfy2 = fl2_list[ix];
      printf("   add(%3d%3d): fl2_list[%3d]",i0+1,i1+1,ix+1);
      for(j=0; j<=nfl2_list_ix; j++){
	if(mdfy2->flip[0] == ix) irev = 1;
	else irev = 0;
	printf("->%3d(%2d)[%d]", mdfy2->flip[irev]+1,
	       mdfy2->flip[1-irev]+1, mdfy2);
	mdfy2 = mdfy2->fl2_next[irev];
      }printf("\n");
#endif

      /* enumerate necessary 3-flips */
      nfl2_list_ix = nfl2_list[ix];
      mdfy2 = fl2_list[ix];
      for(j=0; j<nfl2_list_ix; j++){
	if(mdfy2->flip[0] == ix) irev = 1;
	else irev = 0;
	i2 = mdfy2->flip[irev];
	mdfy2 = mdfy2->fl2_next[irev];
	if(i2<i0){ ii[0] = i2; ii[1] = i0; ii[2] = i1; }
	else if(i2<i1){ ii[0] = i0; ii[1] = i2; ii[2] = i1; }
	else { ii[0] = i0; ii[1] = i1; ii[2] = i2; }
	add_to_hash(3, ii, 0, &hashvalue, &mdfy_vp, &mdfy3, &mdfy3prev);
	/* add_to_hash(flips = ii, iflip = 3, modify_val = 0);
	 new modify cell is returned to mdfy3 */
	(mdfy3->combpt) ++;
	/* add_to_ring(mdfy3); */
	add_to_ring(hashvalue, mdfy3, mdfy3prev);
      }
    }
    nfl2_list[ix]++;
  }
}

/***** delete unnecessary cells from the candidate ring (pluschk=2,4) ********/
void fl2_delete(struct modify *mdfy)
{
  int i, j, ix, i0, i1, i2, irev, ii[3], nfl2_list_ix, hashvalue, mdfy_vp;
  struct modify *mdfy2, *mdfy_n, *mdfy_p, *mdfy3, *mdfy3prev;

  i0 = mdfy->flip[0];
  i1 = mdfy->flip[1];
  if(i0>i1){printf("order of flips are wrong!\n");exit(1);}/**/
  for(i=0; i<2; i++){
    ix = mdfy->flip[i];
#if FALSE /* debug */
    nfl2_list_ix = nfl2_list[ix];
    mdfy2 = fl2_list[ix];
    printf("delete(%3d%3d): fl2_list[%3d]",i0+1,i1+1,ix+1);
    for(j=0; j<nfl2_list_ix; j++){
      if(mdfy2->flip[0] == ix) irev = 1;
      else irev = 0;
      printf("->%3d", mdfy2->flip[irev]+1);
      mdfy2 = mdfy2->fl2_next[irev];
    }printf("\n");
#endif
    if(nfl2_list[ix] <= 1){
      if(nfl2_list[ix] <= 0){printf("error around fl2_list!\n");exit(1);}/**/
      fl2_list[ix] = NULL;
    }
    else{
      /* delete from fl2_list */
      irev = 1 - i;
      mdfy_n = mdfy->fl2_next[irev];
      mdfy_p = mdfy->fl2_prev[irev];
      if(mdfy_n->flip[0] == ix) irev = 1;
      else irev = 0;
      mdfy_n->fl2_prev[irev] = mdfy_p;
      if(mdfy_p->flip[0] == ix) irev = 1;
      else irev = 0;
      mdfy_p->fl2_next[irev] = mdfy_n;
      if(fl2_list[ix] == mdfy) fl2_list[ix] = mdfy_n;
      /* end of 'delete from fl2_list' */
#if FALSE /* debug */
      nfl2_list_ix = nfl2_list[ix];
      mdfy2 = fl2_list[ix];
      printf("delete(%3d%3d): fl2_list[%3d]",i0+1,i1+1,ix+1);
      for(j=0; j<nfl2_list_ix-1; j++){
	if(mdfy2->flip[0] == ix) irev = 1;
	else irev = 0;
	printf("->%3d(%2d)", mdfy2->flip[irev]+1, mdfy2->flip[1-irev]+1);
	mdfy2 = mdfy2->fl2_next[irev];
      }printf("\n");
#endif

      /* enumerate necessary 3-flips */
      nfl2_list_ix = nfl2_list[ix] - 1;
      mdfy2 = fl2_list[ix];
      for(j=0; j<nfl2_list_ix; j++){
	if(mdfy2->flip[0] == ix) irev = 1;
	else irev = 0;
	i2 = mdfy2->flip[irev];
	mdfy2 = mdfy2->fl2_next[irev];
	if(i2<i0){ ii[0] = i2; ii[1] = i0; ii[2] = i1; }
	else if(i2<i1){ ii[0] = i0; ii[1] = i2; ii[2] = i1; }
	else { ii[0] = i0; ii[1] = i1; ii[2] = i2; }
	add_to_hash(3, ii, 0, &hashvalue, &mdfy_vp, &mdfy3, &mdfy3prev);
	/* add_to_hash(flips = ii, iflip = 3, modify_val = 0);
	 new modify cell is returned to mdfy3 */
	(mdfy3->combpt) --;
	/*if(mdfy3prev!=NULL)
	  printf("next to mdfyprev: %d\n", mdfy3prev->hnext);/**/
	add_to_ring(hashvalue, mdfy3, mdfy3prev);
	/*if(mdfy3prev!=NULL)
	  printf("next to mdfyprev: %d\n", mdfy3prev->hnext);/**/
	/* add_to_ring(mdfy3); */
      }
    }
    nfl2_list[ix]--;
  }
}

/***** free mdfy memoryes after every LS iteration ***************************/
void free_mdfy_mems(void)
{
  int i, hashplus_i;
  struct modify *mdfy;

  for(i=0; i<nhashplus; i++){
    hashplus_i = hashplus[i];
    mdfy = hash[hashplus_i];
    while(mdfy != NULL){
      if(nfree_mems >= free_mems_size){
	free_mems_size += mallocunit;
	if((free_mems=(struct modify **)
	    realloc(free_mems,free_mems_size*sizeof(struct modify *)))
	   ==NULL) exit(1);
      }
      free_mems[nfree_mems] = mdfy;
      nfree_mems++;
      mdfy = mdfy->hnext;
    }
    hash[hashplus_i] = NULL;
  }
  nhashplus = 0;
/*  for(i=0; i<=hashmask; i++){
    mdfy = hash[i];
    while(mdfy != NULL){
      if(nfree_mems >= free_mems_size){
	free_mems_size += mallocunit;
	if((free_mems=(struct modify **)
	    realloc(free_mems,free_mems_size*sizeof(struct modify *)))
	   ==NULL) exit(1);
      }
      free_mems[nfree_mems] = mdfy;
      nfree_mems++;
      mdfy = mdfy->hnext;
    }
    hash[i] = NULL;
  }
  /* hash statistics */
  sum_max_nmem_used += max_nmem_used;
  sum_max_hash_trace += max_hash_trace;
#if FALSE
  printf("# max_mem_used =%6d  max_hash_trace =%4d (ideal: %3d)\n",
	 max_nmem_used, max_hash_trace, (int)max_nmem_used/(1<<hashsize)+1);
#endif
  nmem_used = max_nmem_used = max_hash_trace = 0;
}

/***** initialize inzeroclause ***********************************************/
void init_inzeroclause(void)
{
  int i, j, nlit_i, lit_ij;

  nzeroclause = 0;
  for(i=0; i<n; i++) inzeroclause[i]=0;
  for(i=0; i<m; i++) if(ntrue[i]==0){
    nzeroclause++;
    nlit_i = nlit[i];
    for(j=0; j<nlit_i; j++){
      lit_ij = lit[i][j];
      lit_ij = ABS(lit_ij) - 1;
      /*printf("%d %d\n", lit[i][j], lit_ij);/**/
      inzeroclause[lit_ij]++;
    }
  }
}

/***** check if there exists a variable included in clause with ntrue=0 ******/
int included_in_zerocl(int iflip, int *ii)
{
  int i;
  for(i=0; i<iflip; i++) if(inzeroclause[ii[i]] > 0) return(1);
  return(0);
}

/***** greatest common divisor ***********************************************/
int gcd00(int aa, int bb)
{
  int dd, temp;

  /*printf(" GCD of %d and %d", aa, bb);/**/
  if(bb > aa){
    temp = bb;
    bb = aa;
    aa = temp;}
  while(bb > 0){
    dd = aa % bb;
    aa = bb;
    bb = dd;
    /*printf("aa = %d, bb = %d, dd = %d\n", aa, bb, dd);/**/
  }
  /*printf(" is %d\n", aa);/**/
  return(aa);
}

/***** main ******************************************************************/
int main(int argc, char *argv[])
{
  int i;

  read_parameters(argc, argv);
  read_data();
  output_parameters();

  starttime=cpu_time(); /*clock()/1000000.0;*/

  init_mems();
  x_n=seed;
  bestgain = -INT_MAX;
  isamples=imoves=ntry=optfound=nmodify_ring_sum=0;
  sum_memory_access_calc=sum_memory_access_update=0;
  sum_gain=0;

  if(maxmoves==0) output_current_state();
  else output_current_state_m();
  output_current_state_mem();
  /**** make neighbor order list ****/
  if(pluschk==0) make_nb_order_list();

  for(i=0; ( cpu_time()-starttime < timelim ) &&
           (
            (maxmemacc>0 &&
	     (sum_memory_access_calc+sum_memory_access_update)<maxmemacc) ||
            (maxmemacc==0 && itry>0 &&ntry<itry) ||
            (maxmemacc==0 && itry==0 && imoves<maxmoves) ||
            (maxmemacc==0 && itry==0 &&maxmoves==0 && isamples<maxsamples)
	   );i++){
    ntry++;
    random_init();
    if(cgain>bestgain){bestgain = cgain; update_bestsol();}
    if(maxmoves==0) output_current_state();
    improve_csol();
    sum_gain += cgain;
    if(optfound==1)if(outelo==0 && optchk==1)break;
    if(initrev==2){
      ntry++;
      swap_csol_and_csol_rev();
      if(cgain>bestgain){bestgain = cgain; update_bestsol();}
      if(maxmoves==0) output_current_state();
      improve_csol();
      sum_gain += cgain;
      if(optfound==1)if(outelo==0 && optchk==1)break;
    }
  }

  output_results();
  output_HJ_lb();
  free_memories();
}
